%%%----------------------------------------------------------------------
%%% File    : mod_passrecover.erl
%%% Author  : Badlop
%%% Purpose : Password recovery using external email account
%%% Created : 
%%% Id      : 0.2.1 Added JID conversion to utf-8
%%%----------------------------------------------------------------------

%%% INSTALL: 
%%%  1 Copy mod_passrecover.erl epop.erl epop_client.erl smtp_fsm.erl to ejabberd/src/
%%%  2 Recompile ejabberd
%%%  3 Create one (optionally two) email address on the local email server (if you have one)
%%%      or on a remote email server.
%%%  4 Add to ejabberd.cfg, 'modules' section the basic configuration:
%%%     {mod_passrecover, [{pop3_password, "F3sNk50"},}, {smtp_password, "LP9h4Z4M"}
%%%                        {smtp_email, "pwdresponder@server.com"}],

%%% CONFIGURE:
%%%
%%%   REQUIRED:
%%%     pop3_password: Password for the POP3 login
%%%     smtp_password: Password for the SMTP login
%%%     smtp_email: Email address to send the message
%%%
%%%   OPTIONAL:
%%%     pop3_login: Login to use for the POP3 server (default: "passrecover@localhost")
%%%     smtp_server: IP address of the SMTP server (default: "127.0.0.1")
%%%     smtp_user: Username to log on the SMTP server (default: "pwdresponder")
%%%     interval: Time between each mailbox check, in minutes (default: 30)
%%%     purge_hours: Maximum hours that the user can take to answer the Request Confirmation (default: 24)
%%%     dumpdir: Directory were erroneous emails are dumped to file (default: "/tmp/")

%%% EXAMPLE CONFIGURATION:
%%%   {mod_passrecover, [{pop3_login, "passrecover@localhost"}, {pop3_password, "F3sNk50"},
%%%                  {smtp_server, "127.0.0.1"}, {smtp_user, "pwdresponder"}, 
%%%                  {smtp_password, "LP9h4Z4M"}, {smtp_email, "pwdresponder@server.com"},
%%%                  {interval, 15}, {purge_hours, 12}, {dumpdir, "/var/log/ejabberd/passrecov/"}]},

%%% USAGE:
%%%  1 You must set your email on the vCard before losing the password. Example email address: 'bob@myserver.net'.
%%%  2 If you forget the password of your Jabber account 'bob@jabber.myserver.net', send this message:
%%%     To: <passrecover@server.com>
%%%     From: <bob@myserver.net>
%%%     Subject: PASSWORD REQUEST
%%%
%%%     JID: bob@jabber.myserver.net
%%%  3 After some minutes, you will receive an email on your account if you are the real owner of that JID.
%%%    The message includes a Validation Code. Simply send back this email to the sender.
%%%  4 After some minutes you will receive an email with the new password.

-module(mod_passrecover).

-behaviour(gen_mod).

-export([start/2,
	 stop/1,
	 send_email/3,
	 loop/3]).

-include("ejabberd.hrl").
-include("jlib.hrl").

-record(passrecover, {validationcode, jid, email, timestamp}).
-record(options, {dumpdir, pop3_login, pop3_password, smtp_server, smtp_user, smtp_password, smtp_email}).

-define(PROCNAME, ejabberd_mod_passrecover).
-define(PASSREQU, "PASSWORD REQUEST").
-define(PASSVALI, "REQUEST CONFIRMATION").
-define(NEWPASS,  "NEW PASSWORD").
-define(VALICODE, "Validation Code").


%%---------------------
%% Module interface
%%---------------------

start(_Host, Opts) ->
	case whereis(?PROCNAME) of
		undefined ->
			Interval = gen_mod:get_opt(interval, Opts, 30), % in minutes
			I = Interval*60*1000,

			Purge_hours = gen_mod:get_opt(purge_hours, Opts, 24),

			Options = #options{
				dumpdir = gen_mod:get_opt(dumpdir, Opts, "/tmp/"),
				pop3_login = gen_mod:get_opt(pop3_login, Opts, "passrecover@localhost"),
				pop3_password = gen_mod:get_opt(pop3_password, Opts, ""),
				smtp_server = gen_mod:get_opt(smtp_server, Opts, "127.0.0.1"),
				smtp_user = gen_mod:get_opt(smtp_user, Opts, "pwdresponder"),
				smtp_password = gen_mod:get_opt(smtp_password, Opts, ""),
				smtp_email = "<"++gen_mod:get_opt(smtp_email, Opts, "")++">"
			},

			ets:new(passrecover, [named_table, public, {keypos, 2}]),

			register(?PROCNAME, spawn(?MODULE, loop, [I, Purge_hours, Options]));
		_ ->
			ok
	end.

loop(I, Purge_hours, Options) ->
	purge_old_requests(Purge_hours),
	Purge_after = 24,
	look_for_emails(Options),
	timer:send_after(I, look),
	receive
		{get_opts, Pid} -> 
			Pid ! Options,
			?MODULE:loop(I, Purge_hours, Options);
		look -> 
			?MODULE:loop(I, Purge_hours, Options);
		stop -> ok
	end.

stop(_Host) ->
    ets:delete(passrecover),
    exit(whereis(?PROCNAME), stop),
    {wait, ?PROCNAME}.


%%---------------------
%% General
%%---------------------

look_for_emails(O) ->
	POP3_login = O#options.pop3_login,
	POP3_password = O#options.pop3_password,
	DumpDir = O#options.dumpdir,
	{ok, SK} = epop_client:connect(POP3_login, POP3_password),
	Scan = epop_client:scan(SK),
	case Scan of
		{ok, []} -> no_emails;
		{ok, Messages} -> process_messages(Messages, SK, DumpDir, O)
	end,
	epop_client:quit(SK).

process_messages(Messages, SK, DumpDir, O) ->
	F = fun({Email_n, _Size}) -> 
		{ok, Email_string} = epop_client:retrieve(SK, Email_n),
		ok = epop_client:delete(SK, Email_n),
		Email = prepare_email(Email_string),
		case check_type(Email) of
			{error, Error_message} -> 
				dump_email(Error_message, Email_string, DumpDir);
			Res -> process_message(Res, O)
		end
	end,
	lists:foreach(F, Messages).

prepare_email(Email_string) ->
	{ok, Email_list_string} = regexp:split(Email_string, "\r\n"),
	lists:map(fun(S)->regexp:split(S,": ") end, Email_list_string).

check_type(Email) -> 
	From_raw = get_elem(Email, "From", header),
	{match, Start, Length} = regexp:match(From_raw, "[a-zA-Z0-9_\\.\-]+@[a-zA-Z0-9\\.\-]+"),
	From1 = string_lowercase(string:substr(From_raw, Start, Length)),
	From = From1,
	
	case check_subject(Email) of
		password_request -> 
			case check_jid_email(Email, From) of
				{error, E} -> {error, E};
				{ok, Jid} -> {password_request, Jid, From1}
			end;
		password_validate ->
			case check_validationcode(Email, From1) of
				{error, E} -> {error, E};
				{ok, Jid} -> {password_validate, Jid, From1}
			end;
		{error, E} -> {error, E}; _ -> {error, "Unknown error"}
	end.

check_subject(Email) -> check_subject2(get_elem(Email, "Subject", header)).
check_subject2(?PASSREQU) -> password_request;
check_subject2(?PASSVALI) -> password_validate;
check_subject2(Subject) ->
	case string:str(Subject, ?PASSVALI) of
		0 -> {error, "Wrong subject on email"};
		_ -> password_validate
	end.

process_message({password_request, Jid, Email_address}, O) -> 
	To = Email_address,
	Subject = ?PASSVALI,
	Jid_string = jlib:jid_to_string(Jid),
	ValidationCode = randoms:get_string(),
	Content = "Somebody has requested to change the password on this Jabber account: "++Jid_string++" \
To get a new password you must confirmate your request: simply reply to this email. \
If you don't want to change your password, ignore this email. \
\
" ++ ?VALICODE ++ ": " ++ ValidationCode,
	store_validation_code(Jid, To, ValidationCode),
	send_email(To, Subject, Content, O);

process_message({password_validate, Jid, Email_address}, O) -> 
	NewPassword = randoms:get_string(),
	ejabberd_auth:set_password(Jid#jid.user, Jid#jid.server, NewPassword),
	To = Email_address,
	Subject = ?NEWPASS,
	Jid_string = jlib:jid_to_string(Jid),
	Content = "The new password for your Jabber account "++Jid_string++" is: "++NewPassword,
	send_email(To, Subject, Content, O);

process_message(_Res, _O) -> 
	ok.


%%---------------------
%% First step
%%---------------------

check_jid_email(Email, From) -> 
	case get_elem(Email, "JID", relaxed) of
		none -> 
			{error, "JID not provided"};
		Jid_string -> 
			Jid = jlib:string_to_jid(iconv:convert("", "utf-8", Jid_string)),
			case check_jid_exists(Jid) of
				false -> {error, "user does not exist"};
				{error, E} -> {error, E};
				true -> 
					case get_jid_email(Jid) of
						{ok, ""} -> {error, "email not available on vcard"};
						{ok, Vcard_email} ->
							case string:equal(Vcard_email, From) of
								true -> {ok, Jid};
								false -> {error, "email address does not match vcard email"}
							end
					end
			end
	end.

check_jid_exists(error) -> {error, "JID on email could not be converted to UTF-8"};
check_jid_exists(Jid) -> 
	ejabberd_auth:is_user_exists(Jid#jid.user, Jid#jid.server).

get_jid_email(Jid) ->
    [{_, _, A1}] = mnesia:dirty_read(vcard, {Jid#jid.user, Jid#jid.server}),
    A2 = xml:get_subtag(A1, "EMAIL"),
    A3 = xml:get_subtag(A2, "USERID"),
    Elem = case A3 of
		"" -> A2;
		_ -> A3
    end,
	Email = string_lowercase(xml:get_tag_cdata(Elem)),
	{ok, Email}.

string_lowercase(String) ->
	stringprep:tolower(String).


%%---------------------
%% Second step
%%---------------------

check_validationcode(Email, From) ->
	ValidationCode = get_elem(Email, ?VALICODE, relaxed),
	case retrieve_validation_code(ValidationCode) of
		{error, E} -> {error, E};
		PR -> 
			case From == PR#passrecover.email of
				true -> 
					delete_validation_code(ValidationCode),
					{ok, PR#passrecover.jid};
				false -> 
					{error, "The email on the valiation does not match the validation code"}
			end
	end.

%%---------------------
%% Utilities
%%---------------------

get_elem(L, E, Type) ->
	F = fun(X, Y) -> 
		case Type of
			header -> X==Y;
			relaxed -> string:str(X, Y)>0
		end
	end,
	Res = [B || {ok, [A | B]} <- L, F(A, E)],
	case lists:flatten(Res) of
		[] -> none;
		R -> R
	end.

send_email(To, Subject, Content) ->
	?PROCNAME ! {get_opts, self()},
	receive O -> O end,
	send_email(To, Subject, Content, O).

send_email(To, Subject, Content, O) ->
	SMTP_server = O#options.smtp_server,
	SMTP_user = O#options.smtp_user,
	SMTP_password = O#options.smtp_password,
	SMTP_email = O#options.smtp_email,

	Msg = simp_msg(SMTP_email, To, Subject, Content),

	{ok,Pid} = smtp_fsm:start(SMTP_server),
	smtp_fsm:ehlo(Pid),
	smtp_fsm:features(Pid),
	smtp_fsm:login(Pid, SMTP_user, SMTP_password),
	?INFO_MSG("The following ERROR REPORT and CRASH REPORT are ficticious, and are generated by ejabberd/mod_recoverpass/smtp_fsm. Don't worry about them.", []),
	ok = smtp_fsm:sendemail(Pid, SMTP_email, "<"++To++">", Msg),
	smtp_fsm:close(Pid).

simp_msg(From, To, Subject, Message) ->
	FromStr = ["From: ", From, "\r\n"],
	ToStr = ["To: ", To, "\r\n"],
	SubjStr = ["Subject: ", Subject, "\r\n"],
	MsgStr = ["\r\n", Message],
	lists:concat(lists:concat([FromStr, ToStr, SubjStr, MsgStr, ["\r\n"]])).

store_validation_code(Jid, To, ValidationCode) ->
	{{Y, M, D}, {H, _, _}} = calendar:now_to_universal_time(now()),
	Timestamp = (Y*365 + M*31 + D)*24 + H,
	P = #passrecover{jid = Jid, email = To, timestamp = Timestamp, validationcode = ValidationCode},
	ets:insert(passrecover, P).

retrieve_validation_code(ValidationCode) ->
	case ets:lookup(passrecover, ValidationCode) of
		[PR] -> PR;
		[] -> {error, "The validation code on the email is not on the database"}
	end.

delete_validation_code(ValidationCode) ->
	ets:delete(passrecover, ValidationCode).

dump_email(Error_message, Email_string, DumpDir) ->
	Timestamp = jlib:timestamp_to_iso(calendar:now_to_universal_time(now())),
	DumpFile = filename:join([DumpDir, string:concat(Timestamp, ".txt")]),
	{ok, F} = file:open(DumpFile, [append]),
	file:write(F, io_lib:format("~n~n ---- Email rejected, reason: ~s~n~n~s", [Error_message, Email_string])),
	file:close(F).

purge_old_requests(Purge_after) ->
	{{Y, M, D}, {H, _, _}} = calendar:now_to_universal_time(now()),
	Timestamp = (Y*365 + M*31 + D)*24 + H,
	Max_hours = Timestamp - Purge_after,
	Sel = [{#passrecover{timestamp = '$1', email = '$2', _ = '_'},
		[{'<', '$1', Max_hours}],
		['$2']
	}],
	ets:select_delete(passrecover, Sel).
